package com.clx.performance.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.ParseException;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.SocketTimeoutException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Slf4j
public class HttpUtils {

        private static PoolingHttpClientConnectionManager connectionManager;
        private static String DEFAULT_STR = "";
        private static String UTF_8 = "UTF-8";
        private static final int CONNECT_TIMEOUT = 10000;
        private static final int SOCKET_TIMEOUT = 100000;
        private static final int REQUESTCONNECT_TIMEOUT = 2000;
        private static final int MAX_TOTAL = 50;
        private static final int CONNECT_DEFAULT_ROUTE = 5;


        /**
         * get
         *
         * @param host
         * @param path
         * @param headers
         * @param querys
         * @return
         * @throws Exception
         */
        public static HttpResponse doGet(String host, String path,
                                         Map<String, String> headers,
                                         Map<String, String> querys)
                throws Exception {
            HttpClient httpClient = wrapClient(host);

            HttpGet request = new HttpGet(buildUrl(host, path, querys));
            for (Map.Entry<String, String> e : headers.entrySet()) {
                request.addHeader(e.getKey(), e.getValue());
            }

            return httpClient.execute(request);
        }

        /**
         * post form
         *
         * @param host
         * @param path
         * @param headers
         * @param querys
         * @param bodys
         * @return
         * @throws Exception
         */
        public static HttpResponse doPost(String host, String path,
                                          Map<String, String> headers,
                                          Map<String, String> querys,
                                          Map<String, String> bodys)
                throws Exception {
            HttpClient httpClient = wrapClient(host);

            HttpPost request = new HttpPost(buildUrl(host, path, querys));
            for (Map.Entry<String, String> e : headers.entrySet()) {
                request.addHeader(e.getKey(), e.getValue());
            }

            if (bodys != null) {
                List<NameValuePair> nameValuePairList = new ArrayList<NameValuePair>();

                for (String key : bodys.keySet()) {
                    nameValuePairList.add(new BasicNameValuePair(key, bodys.get(key)));
                }
                UrlEncodedFormEntity formEntity = new UrlEncodedFormEntity(nameValuePairList, "utf-8");
                formEntity.setContentType("application/x-www-form-urlencoded; charset=UTF-8");
                request.setEntity(formEntity);
            }

            return httpClient.execute(request);
        }

        /**
         * Post String
         *
         * @param host
         * @param path
         * @param headers
         * @param querys
         * @param body
         * @return
         * @throws Exception
         */
        public static HttpResponse doPost(String host, String path,
                                          Map<String, String> headers,
                                          Map<String, String> querys,
                                          String body)
                throws Exception {
            HttpClient httpClient = wrapClient(host);

            HttpPost request = new HttpPost(buildUrl(host, path, querys));
            for (Map.Entry<String, String> e : headers.entrySet()) {
                request.addHeader(e.getKey(), e.getValue());
            }

            if (StringUtils.isNotBlank(body)) {
                request.setEntity(new StringEntity(body, "utf-8"));
            }

            return httpClient.execute(request);
        }

        /**
         * Post stream
         *
         * @param host
         * @param path
         * @param headers
         * @param querys
         * @param body
         * @return
         * @throws Exception
         */
        public static HttpResponse doPost(String host, String path,
                                          Map<String, String> headers,
                                          Map<String, String> querys,
                                          byte[] body)
                throws Exception {
            HttpClient httpClient = wrapClient(host);

            HttpPost request = new HttpPost(buildUrl(host, path, querys));
            for (Map.Entry<String, String> e : headers.entrySet()) {
                request.addHeader(e.getKey(), e.getValue());
            }

            if (body != null) {
                request.setEntity(new ByteArrayEntity(body));
            }

            return httpClient.execute(request);
        }

        /**
         * Put String
         *
         * @param host
         * @param path
         * @param headers
         * @param querys
         * @param body
         * @return
         * @throws Exception
         */
        public static HttpResponse doPut(String host, String path,
                                         Map<String, String> headers,
                                         Map<String, String> querys,
                                         String body)
                throws Exception {
            HttpClient httpClient = wrapClient(host);

            HttpPut request = new HttpPut(buildUrl(host, path, querys));
            for (Map.Entry<String, String> e : headers.entrySet()) {
                request.addHeader(e.getKey(), e.getValue());
            }

            if (StringUtils.isNotBlank(body)) {
                request.setEntity(new StringEntity(body, "utf-8"));
            }

            return httpClient.execute(request);
        }

        /**
         * Put stream
         *
         * @param host
         * @param path
         * @param headers
         * @param querys
         * @param body
         * @return
         * @throws Exception
         */
        public static HttpResponse doPut(String host, String path,
                                         Map<String, String> headers,
                                         Map<String, String> querys,
                                         byte[] body)
                throws Exception {
            HttpClient httpClient = wrapClient(host);

            HttpPut request = new HttpPut(buildUrl(host, path, querys));
            for (Map.Entry<String, String> e : headers.entrySet()) {
                request.addHeader(e.getKey(), e.getValue());
            }

            if (body != null) {
                request.setEntity(new ByteArrayEntity(body));
            }

            return httpClient.execute(request);
        }

        /**
         * Delete
         *
         * @param host
         * @param path
         * @param headers
         * @param querys
         * @return
         * @throws Exception
         */
        public static HttpResponse doDelete(String host, String path,
                                            Map<String, String> headers,
                                            Map<String, String> querys)
                throws Exception {
            HttpClient httpClient = wrapClient(host);

            HttpDelete request = new HttpDelete(buildUrl(host, path, querys));
            for (Map.Entry<String, String> e : headers.entrySet()) {
                request.addHeader(e.getKey(), e.getValue());
            }

            return httpClient.execute(request);
        }

        private static String buildUrl(String host, String path, Map<String, String> querys) throws UnsupportedEncodingException {
            StringBuilder sbUrl = new StringBuilder();
            sbUrl.append(host);
            if (!StringUtils.isBlank(path)) {
                sbUrl.append(path);
            }
            if (null != querys) {
                StringBuilder sbQuery = new StringBuilder();
                for (Map.Entry<String, String> query : querys.entrySet()) {
                    if (0 < sbQuery.length()) {
                        sbQuery.append("&");
                    }
                    if (StringUtils.isBlank(query.getKey()) && !StringUtils.isBlank(query.getValue())) {
                        sbQuery.append(query.getValue());
                    }
                    if (!StringUtils.isBlank(query.getKey())) {
                        sbQuery.append(query.getKey());
                        if (!StringUtils.isBlank(query.getValue())) {
                            sbQuery.append("=");
                            sbQuery.append(URLEncoder.encode(query.getValue(), "utf-8"));
                        }
                    }
                }
                if (0 < sbQuery.length()) {
                    sbUrl.append("?").append(sbQuery);
                }
            }

            return sbUrl.toString();
        }

        private static HttpClient wrapClient(String host) {
            HttpClient httpClient = new DefaultHttpClient();
            if (host.startsWith("https://")) {
                sslClient(httpClient);
            }

            return httpClient;
        }

        private static void sslClient(HttpClient httpClient) {
            try {
                SSLContext ctx = SSLContext.getInstance("TLS");
                X509TrustManager tm = new X509TrustManager() {
                    public X509Certificate[] getAcceptedIssuers() {
                        return null;
                    }

                    public void checkClientTrusted(X509Certificate[] xcs, String str) {

                    }

                    public void checkServerTrusted(X509Certificate[] xcs, String str) {

                    }
                };
                ctx.init(null, new TrustManager[]{tm}, null);
                SSLSocketFactory ssf = new SSLSocketFactory(ctx);
                ssf.setHostnameVerifier(SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
                ClientConnectionManager ccm = httpClient.getConnectionManager();
                SchemeRegistry registry = ccm.getSchemeRegistry();
                registry.register(new Scheme("https", 443, ssf));
            } catch (KeyManagementException ex) {
                throw new RuntimeException(ex);
            } catch (NoSuchAlgorithmException ex) {
                throw new RuntimeException(ex);
            }
        }


    public static String httpGetRequest(String url) throws Exception {
        HttpGet httpGet = new HttpGet(url);
        return getResult(httpGet);
    }

    private static void init() {
        if (connectionManager == null) {
            connectionManager = new PoolingHttpClientConnectionManager();
            connectionManager.setMaxTotal(50);
            connectionManager.setDefaultMaxPerRoute(5);
        }

    }

    private static CloseableHttpClient getHttpClient() {
        init();
        RequestConfig.Builder builder = RequestConfig.custom();
        RequestConfig config = builder.setSocketTimeout(100000).setConnectTimeout(10000).setConnectionRequestTimeout(2000).build();
        CloseableHttpClient client = HttpClients.custom().setMaxConnPerRoute(5).disableConnectionState().setDefaultRequestConfig(config).setConnectionManager(connectionManager).build();
        return client;
    }

    private static String getResult(HttpRequestBase request) throws IOException {
        CloseableHttpClient httpClient = getHttpClient();
        CloseableHttpResponse response = null;
        InputStream in = null;

        String var6;
        try {
            response = httpClient.execute(request);
            HttpEntity entity = response.getEntity();
            in = response.getEntity().getContent();
            if (entity == null) {
                return DEFAULT_STR;
            }

            String result = EntityUtils.toString(entity, Charset.forName(UTF_8));
            response.close();
            var6 = result;
        } catch (ConnectTimeoutException var37) {
            log.error("connect timeout {}", var37);
            return DEFAULT_STR;
        } catch (SocketTimeoutException var38) {
            log.error("read timeout {}", var38);
            return DEFAULT_STR;
        } catch (ClientProtocolException var39) {
            log.error("protocol exception {}", var39);
            return DEFAULT_STR;
        } catch (ParseException var40) {
            log.error("parse exception {}", var40);
            return DEFAULT_STR;
        } catch (IOException var41) {
            log.error("network exception {}", var41);
            return DEFAULT_STR;
        } catch (Exception var42) {
            log.error("other exception {}", var42);
            return DEFAULT_STR;
        } finally {
            if (response != null) {
                try {
                    response.close();
                } catch (IOException var36) {
                    var36.printStackTrace();
                }
            }

            if (in != null) {
                try {
                    in.close();
                } catch (IOException var35) {
                    var35.printStackTrace();
                }
            }

        }

        return var6;
    }
}
